package com.vmware.automatic.plugin.registration.services;

import com.vmware.automatic.plugin.registration.resources.CommandLineBuilder;
import com.vmware.vim25.Description;
import com.vmware.vim25.Extension;
import com.vmware.vim25.ExtensionClientInfo;
import com.vmware.vim25.ExtensionResourceInfo;
import com.vmware.vim25.ExtensionServerInfo;
import com.vmware.vim25.KeyValue;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.cli.ParseException;
import org.mockito.Mockito;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PowerMockIgnore;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.testng.PowerMockTestCase;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.GregorianCalendar;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNull;
import static org.testng.Assert.assertTrue;

@PrepareForTest(DatatypeFactory.class)
// Ignore conflict in method's signature between Powermock's classloader and the
// current classloader for "org.apache.commons.cli.MissingArgumentException"
@PowerMockIgnore("org.apache.commons.cli.*")
public class PluginExtensionRegistryServiceTest extends PowerMockTestCase {

   private PluginExtensionRegistryService extensionRegistryService;

   @BeforeClass
   private void init() {
      extensionRegistryService = new PluginExtensionRegistryService();
   }

   // updateTopLevelProperties
   @Test
   public void updateTopLevelProperties_whenExtensionHasNoKey_setsExtensionKey()
         throws Exception {
      // Pre-update extension
      final Extension extension = new Extension();
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder().defaults()
            .build();
      extensionRegistryService.updateTopLevelProperties(extension, commandLine);
      // Asserts
      assertEquals(extension.getKey(), "plugin-key");
   }

   @Test
   public void updateTopLevelProperties_setsProvidedValues()
         throws Exception {
      // Pre-update extension
      final Extension extension = new Extension();
      final String key = "extension-key";
      extension.setKey(key);
      // Command line arguments
      final String version = "test-version";
      final String company = "test-company";
      final boolean showInSolutionManager = true;

      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .version(version)
            .company(company)
            .showInSolutionManager(showInSolutionManager)
            .build();
      extensionRegistryService.updateTopLevelProperties(extension, commandLine);
      // Asserts
      assertEquals(extension.getKey(), key);
      assertEquals(extension.getVersion(), version);
      assertEquals(extension.getCompany(), company);
      assertTrue(extension.isShownInSolutionManager());
   }

   // updateDescription
   @Test
   public void updateDescription_whenDescriptionIsNullAndNoDescriptionMembersProvided_setsDefaultValues()
         throws ParseException {
      // Pre-update extension
      final Extension extension = new Extension();
      // Post update description values
      final String label = "";
      final String summary = "";
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .build();
      extensionRegistryService.updateDescription(extension, commandLine);
      // Asserts
      assertEquals(extension.getDescription().getLabel(), label);
      assertEquals(extension.getDescription().getSummary(), summary);
   }

   @Test
   public void updateDescription_whenDescriptionIsNullAndDescriptionMembersProvided_setsProvidedValues()
         throws ParseException {
      // Pre-update extension
      final Extension extension = new Extension();
      // Post update description values
      final String label = "test label";
      final String summary = "test description";
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .label(label)
            .summary(summary)
            .build();
      extensionRegistryService.updateDescription(extension, commandLine);
      // Asserts
      assertEquals(extension.getDescription().getLabel(), label);
      assertEquals(extension.getDescription().getSummary(), summary);
   }

   @Test
   public void updateDescription_whenDescriptionIsNotNullAndNoDescriptionMembersProvided_keepsExistingValues()
         throws ParseException {
      // Pre-update description values
      final String label = "pre-update test label";
      final String summary = "pre-update test description";
      // Pre-update extension
      final Extension extension = new Extension();
      extension.setDescription(new Description());
      extension.getDescription().setLabel(label);
      extension.getDescription().setSummary(summary);
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .build();

      extensionRegistryService.updateDescription(extension, commandLine);
      // Asserts
      assertEquals(extension.getDescription().getLabel(), label);
      assertEquals(extension.getDescription().getSummary(), summary);
   }

   @Test
   public void updateDescription_whenDescriptionIsNotNullAndDescriptionMembersProvided_setsProvidedValues()
         throws ParseException {
      // Pre-update description values
      final String preUpdateLabel = "pre-update test label";
      final String preUpdateSummary = "pre-update test description";
      // Post update description values
      final String postUpdateLabel = "post-update test label";
      final String postUpdateSummary = "post-update test description";
      // Pre-update extension
      final Extension extension = new Extension();
      extension.setDescription(new Description());
      extension.getDescription().setLabel(preUpdateLabel);
      extension.getDescription().setSummary(preUpdateSummary);
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .label(postUpdateLabel)
            .summary(postUpdateSummary)
            .build();

      extensionRegistryService.updateDescription(extension, commandLine);
      // Asserts
      assertEquals(extension.getDescription().getLabel(), postUpdateLabel);
      assertEquals(extension.getDescription().getSummary(), postUpdateSummary);
   }

   // updateClientInfo
   @Test
   public void updateClientInfo_whenClientInfoIsEmptyAndNoClientInfoMembersProvided_setsDefaultValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      extension.setDescription(new Description());
      extension.getDescription().setSummary("test-summary");
      extension.getDescription().setLabel("test-description");
      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .build();
      extensionRegistryService.updateClientInfo(extension, commandLine);
      // Asserts
      final ExtensionClientInfo clientInfo = extension.getClient().get(0);
      assertEquals(clientInfo.getCompany(), "");
      assertEquals(clientInfo.getType(), "vsphere-client-serenity");
      assertEquals(clientInfo.getDescription().getSummary(), "test-summary");
      assertEquals(clientInfo.getDescription().getLabel(), "test-description");
      assertNull(clientInfo.getUrl());
      assertNull(clientInfo.getVersion());
   }

   @Test
   public void updateClientInfo_whenClientInfoIsEmptyAndClientInfoMembersProvided_setsProvidedValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      final Description description = new Description();
      extension.setDescription(description);
      description.setSummary("test-summary");
      description.setLabel("test-description");

      // Post-update values
      final String postUpdateVersion = "updated-test-version";
      final String postUpdatePluginUrl = "updated-test-plugin-url";

      // Command line arguments
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .version(postUpdateVersion)
            .pluginUrl(postUpdatePluginUrl)
            .remote()
            .build();
      extensionRegistryService.updateClientInfo(extension, commandLine);

      // Asserts
      final ExtensionClientInfo clientInfo = extension.getClient().get(0);
      assertEquals(clientInfo.getCompany(), "");
      assertEquals(clientInfo.getType(), "vsphere-client-remote");
      assertEquals(clientInfo.getDescription().getSummary(),
            description.getSummary());
      assertEquals(clientInfo.getDescription().getLabel(),
            description.getLabel());
      assertEquals(clientInfo.getUrl(), postUpdatePluginUrl);
      assertEquals(clientInfo.getVersion(), postUpdateVersion);
   }

   @Test
   public void updateClientInfo_whenClientInfoIsNotEmptyAndSomeClientInfoMembersProvided_setsProvidedValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      // Description
      extension.setDescription(new Description());
      extension.getDescription().setSummary("test-summary");
      extension.getDescription().setLabel("test-description");
      // ClientInfo
      final ExtensionClientInfo extensionClientInfo = new ExtensionClientInfo();
      extensionClientInfo.setCompany("test-company");
      extensionClientInfo.setDescription(extension.getDescription());
      extensionClientInfo.setType("vsphere-client-serenity");
      extensionClientInfo.setUrl("test-plugin-url");
      extensionClientInfo.setVersion("test-version");
      extension.getClient().add(extensionClientInfo);

      // Post-update values
      final String postUpdateVersion = "updated-test-version";
      final String postUpdatePluginUrl = "updated-test-plugin-url";
      final String postUpdateCompany = "updated-test-company";

      // Command line arguments
      CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .version(postUpdateVersion)
            .pluginUrl(postUpdatePluginUrl)
            .company(postUpdateCompany)
            .build();
      extensionRegistryService.updateClientInfo(extension, commandLine);

      // Asserts
      final ExtensionClientInfo clientInfo = extension.getClient().get(0);
      assertEquals(clientInfo.getType(), "vsphere-client-serenity");
      assertEquals(clientInfo.getDescription().getSummary(), "test-summary");
      assertEquals(clientInfo.getDescription().getLabel(), "test-description");
      assertEquals(clientInfo.getCompany(), postUpdateCompany);
      assertEquals(clientInfo.getUrl(), postUpdatePluginUrl);
      assertEquals(clientInfo.getVersion(), postUpdateVersion);
   }

   @Test
   public void updateClientInfo_whenClientInfoIsNotEmptyAndNoClientInfoMembersProvided_keepsExistingValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      // Description
      extension.setDescription(new Description());
      extension.getDescription().setSummary("test-summary");
      extension.getDescription().setLabel("test-description");
      // ClientInfo
      final String preUpdateCompany = "test-company";
      final String preUpdatePluginUrl = "test-plugin-url";
      final String preUpdateVersion = "test-version";
      final ExtensionClientInfo extensionClientInfo = new ExtensionClientInfo();
      extensionClientInfo.setDescription(extension.getDescription());
      extensionClientInfo.setType("vsphere-client-serenity");
      extensionClientInfo.setCompany("test-company");
      extensionClientInfo.setUrl("test-plugin-url");
      extensionClientInfo.setVersion("test-version");
      extension.getClient().add(extensionClientInfo);

      // Command line arguments
      CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .build();
      extensionRegistryService.updateClientInfo(extension, commandLine);

      // Asserts
      final ExtensionClientInfo clientInfo = extension.getClient().get(0);
      assertEquals(clientInfo.getType(), "vsphere-client-serenity");
      assertEquals(clientInfo.getDescription().getSummary(), "test-summary");
      assertEquals(clientInfo.getDescription().getLabel(), "test-description");
      assertEquals(clientInfo.getCompany(), preUpdateCompany);
      assertEquals(clientInfo.getUrl(), preUpdatePluginUrl);
      assertEquals(clientInfo.getVersion(), preUpdateVersion);
   }

   // update resourceInfo
   @Test
   public void updateResourceInfo_whenResourceInfoIsEmptyAndNoResourceInfoMembersProvided_setsDefaults()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();

      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .build();

      extensionRegistryService.updateResourceInfo(extension, commadLine);

      // Asserts
      assertEquals(extension.getResourceList().size(), 1);
      final ExtensionResourceInfo resourceInfo = extension.getResourceList()
            .get(0);
      assertEquals(resourceInfo.getLocale(),"en_US");
      assertEquals(resourceInfo.getModule(),"name");
      assertEquals(resourceInfo.getData().size(),1);
      final KeyValue data = resourceInfo.getData().get(0);
      assertEquals(data.getKey(), "name");
      assertEquals(data.getValue(), "");
   }

   @Test
   public void updateResourceInfo_whenResourceInfoIsEmptyAndResourceInfoMembersProvided_setsProvidedValues()
         throws ParseException {
      final Extension extension = new Extension();
      final String postUpdateName = "updated-test-name";
      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .label(postUpdateName)
            .build();

      extensionRegistryService.updateResourceInfo(extension, commadLine);

      // Asserts
      assertEquals(extension.getResourceList().size(), 1);
      final ExtensionResourceInfo resourceInfo = extension.getResourceList()
            .get(0);
      assertEquals(resourceInfo.getLocale(),"en_US");
      assertEquals(resourceInfo.getModule(),"name");
      assertEquals(resourceInfo.getData().size(),1);
      final KeyValue data = resourceInfo.getData().get(0);
      assertEquals(data.getKey(), "name");
      assertEquals(data.getValue(), postUpdateName);
   }

   @Test
   public void updateResourceInfo_whenResourceInfoIsNotEmptyAndResourceInfoMembersProvided_setsProvidedValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      final ExtensionResourceInfo preUpdateResourceInfo =
            new ExtensionResourceInfo();
      extension.getResourceList().add(preUpdateResourceInfo);
      preUpdateResourceInfo.getData().add(new KeyValue());
      preUpdateResourceInfo.getData().get(0).setKey("name");
      preUpdateResourceInfo.getData().get(0).setValue("value");

      // Command line arguments
      final String postUpdateName = "updated-test-name";
      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .label(postUpdateName)
            .build();

      extensionRegistryService.updateResourceInfo(extension, commadLine);

      // Asserts
      assertEquals(extension.getResourceList().size(), 1);
      final ExtensionResourceInfo resourceInfo = extension.getResourceList()
            .get(0);
      assertEquals(resourceInfo.getData().size(),1);
      final KeyValue data = resourceInfo.getData().get(0);
      assertEquals(data.getKey(), "name");
      assertEquals(data.getValue(), postUpdateName);
   }

   @Test
   public void updateResourceInfo_whenResourceInfoIsNotEmptyAndNoResourceInfoMembersProvided_keepsExistingValues()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      final ExtensionResourceInfo preUpdateResourceInfo =
            new ExtensionResourceInfo();
      extension.getResourceList().add(preUpdateResourceInfo);
      preUpdateResourceInfo.getData().add(new KeyValue());
      final String preUpdateKey = "test-name";
      final String preUpadteValue = "test-value";
      preUpdateResourceInfo.getData().get(0).setKey(preUpdateKey);
      preUpdateResourceInfo.getData().get(0).setValue(preUpadteValue);

      // Command line arguments
      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .build();

      extensionRegistryService.updateResourceInfo(extension, commadLine);

      // Asserts
      assertEquals(extension.getResourceList().size(), 1);
      final ExtensionResourceInfo resourceInfo = extension.getResourceList()
            .get(0);
      assertEquals(resourceInfo.getData().size(),1);
      final KeyValue data = resourceInfo.getData().get(0);
      assertEquals(data.getKey(), preUpdateKey);
      assertEquals(data.getValue(), preUpadteValue);
   }

   // updateServerInfo
   @Test
   public void updateServerInfo_whenServerInfoIsEmptyAndProvidedServerInfoIsHTTP_doesNotSetAnything()
         throws ParseException {
      final Extension extension = new Extension();
      // Set system's out in order to test INFO message
      final ByteArrayOutputStream printedContent = new ByteArrayOutputStream();
      System.setOut(new PrintStream(printedContent));
      // Expected INFO message
      final String expectedInfoMsg =
            "INFO: Not using https for your plugin URL is OK for testing but not recommended for production."
                  + "\nUsers will have to include the flag allowHttp=true in their vSphere Client webclient.properties otherwise the http URL will be ignored"
                  + System.lineSeparator();
      final String postUpdateHttpUrl = "http://test-plugin-url.com";
      CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .pluginUrl(postUpdateHttpUrl)
            .build();
      extensionRegistryService.updateServerInfo(extension, commadLine);
      assertEquals(extension.getServer().size(), 0);
      assertEquals(printedContent.toString(), expectedInfoMsg);
      // Restore original system's out
      System.setOut(System.out);
   }

   @Test
   public void updateServerInfo_whenServerInfoIsHTTPSAndNoServerInfoMembersProvided_doesNotSetAnything()
         throws ParseException {
      final Extension extension = new Extension();
      final ExtensionServerInfo serverInfo = new ExtensionServerInfo();
      serverInfo.setUrl("https://existing-test-url");
      serverInfo.setType("HTTPS");
      extension.getServer().add(serverInfo);

      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .build();
      extensionRegistryService.updateServerInfo(extension, commadLine);

      assertEquals(extension.getServer().size(), 1);
   }

   @Test
   public void updateServerInfo_whenServerInfoIsEmptyAndProvidedServerInfoIsHTTPS_setsCorrectValues()
         throws ParseException {
      Extension extension = new Extension();

      // Post-update values
      final String postUpdateHttpsUrl = "https://test-url.com";
      final String postUpdateServerThumbprint = "test:server:thumbprint";

      CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .pluginUrl(postUpdateHttpsUrl)
            .serverThumbprint(postUpdateServerThumbprint)
            .build();
      extensionRegistryService.updateServerInfo(extension, commandLine);

      // Asserts
      assertEquals(extension.getServer().size(), 1);
      final ExtensionServerInfo serverInfo = extension.getServer().get(0);
      assertEquals(serverInfo.getAdminEmail().size(),1);
      assertEquals(serverInfo.getAdminEmail().get(0), "noreply@vmware.com");
      assertEquals(serverInfo.getType(), "HTTPS");
      assertEquals(serverInfo.getCompany(), "");
      assertNull(serverInfo.getDescription());
      assertEquals(serverInfo.getUrl(), postUpdateHttpsUrl);
      assertEquals(serverInfo.getServerThumbprint(), postUpdateServerThumbprint);
   }

   @Test(expectedExceptions = { MissingArgumentException.class })
   public void updateServerInfo_whenServerInfoIsEmptyAndProvidedServerInfoIsHTTPSButNoServerThumbprintProvided_throwsException()
         throws ParseException {
      final Extension extension = new Extension();

      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .pluginUrl("https://test-plugin-url")
            .build();
      extensionRegistryService.updateServerInfo(extension, commandLine);
   }

   @Test
   public void updateServerInfo_whenServerInfoIsNotEmptyAndProvidedServerInfoIsHTTPSButNoServerThumbprintProvided_updatesProvidedValuesOnly()
         throws ParseException {
      // Pre-update values
      final Extension extension = new Extension();
      final ExtensionServerInfo preUpdateServerInfo = new ExtensionServerInfo();
      final String preUpdateServerThumbprint = "pre-update-thumbprint";
      final String preUpdateAdminEmail = "admin@yahoo.com";
      final String preUpdatePluginUrl = "https://pre-update-url";
      final String preUpdateLabel = "pre-update-label";
      final String preUpdateSummary = "pre-update-summary";
      final String preUpdateType = "HTTPS";
      preUpdateServerInfo.setCompany("");
      preUpdateServerInfo.setType(preUpdateType);
      preUpdateServerInfo.getAdminEmail().add(preUpdateAdminEmail);
      preUpdateServerInfo.setUrl(preUpdatePluginUrl);
      preUpdateServerInfo.setServerThumbprint(preUpdateServerThumbprint);
      extension.setDescription(new Description());
      preUpdateServerInfo.setDescription(extension.getDescription());
      preUpdateServerInfo.getDescription().setLabel(preUpdateLabel);
      preUpdateServerInfo.getDescription().setSummary(preUpdateSummary);
      extension.getServer().add(preUpdateServerInfo);

      // Post-update values
      final String postUpdateCompany = "post-update-company";
      final CommandLine commandLine = new CommandLineBuilder()
            .defaults()
            .company(postUpdateCompany)
            .build();
      extensionRegistryService.updateServerInfo(extension, commandLine);

      // Asserts
      assertEquals(extension.getServer().size(), 1);
      final ExtensionServerInfo postUpdateServerInfo = extension.getServer()
            .get(0);
      assertEquals(postUpdateServerInfo.getServerThumbprint(),
            preUpdateServerThumbprint);
      assertEquals(postUpdateServerInfo.getAdminEmail().get(0),
            preUpdateAdminEmail);
      assertEquals(postUpdateServerInfo.getUrl(), preUpdatePluginUrl);
      assertEquals(postUpdateServerInfo.getType(), preUpdateType);
      assertEquals(postUpdateServerInfo.getDescription().getLabel(),
            preUpdateLabel);
      assertEquals(postUpdateServerInfo.getDescription().getSummary(),
            preUpdateSummary);
      assertEquals(postUpdateServerInfo.getCompany(), postUpdateCompany);
   }

   @Test
   public void updateServerInfo_whenServerInfoIsHTTPSAndProvidedPluginUrlIsHTTP_unsetsServerInfo()
         throws ParseException {
      final Extension extension = new Extension();
      final ExtensionServerInfo serverInfo = new ExtensionServerInfo();
      serverInfo.setUrl("https://existing-test-url");
      serverInfo.setType("HTTPS");
      extension.getServer().add(serverInfo);

      final CommandLine commadLine = new CommandLineBuilder()
            .defaults()
            .pluginUrl("http://new-plugin-url")
            .company("test-company")
            .build();
      extensionRegistryService.updateServerInfo(extension, commadLine);

      assertEquals(extension.getServer().size(), 0);
   }

   @Test
   public void updateCalendarInfo_setsLastModificationTime()
         throws DatatypeConfigurationException {
      // Create calendar mocks
      PowerMockito.mockStatic(DatatypeFactory.class);
      DatatypeFactory dtFactoryMock = PowerMockito.mock(DatatypeFactory.class);
      XMLGregorianCalendar xmlCalendarMock = PowerMockito
            .mock(XMLGregorianCalendar.class);
      // Expect mocks
      Mockito.when(DatatypeFactory.newInstance()).thenReturn(dtFactoryMock);
      Mockito.when(dtFactoryMock
            .newXMLGregorianCalendar(Mockito.any(GregorianCalendar.class)))
            .thenReturn(xmlCalendarMock);

      // Execute method
      final Extension extension = new Extension();
      extensionRegistryService.updatelastHeartbeatTime(extension);

      // Assert lastHeartbeatTime
      assertEquals(extension.getLastHeartbeatTime(), xmlCalendarMock);
   }
}
